<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class FelInvoice_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /*
     * ==================================================
     * FLUJO REAL CORRECTO
     * ==================================================
     * 1️⃣ obtener venta
     * 2️⃣ obtener token FEL (certificar)
     * 3️⃣ obtener token NUC (consultar NIT)
     * 4️⃣ consultar NIT
     * 5️⃣ construir JSON
     * 6️⃣ certificar
     * 7️⃣ guardar
     * NUNCA certificar antes del NIT
     * ==================================================
     */
    public function emitirDesdeSale($reference_no, $nit, $tipo = 'FACT', $modo_prueba = true, $taxid = null, $username = null)
    {
        /*
         * ===============================
         * 1️⃣ OBTENER VENTA
         * ===============================
         */
        $sale = $this->db
            ->where('reference_no', $reference_no)
            ->get('sma_sales')
            ->row_array();

        if (!$sale) {
            throw new Exception('Venta no encontrada');
        }

        $items = $this->db
            ->where('sale_id', $sale['id'])
            ->get('sma_sale_items')
            ->result_array();

        /*
         * ===============================
         * 2️⃣ CONEXIÓN CERTIFICADOR
         * ===============================
         */
        $this->load->admin_model('Certificadores_model', 'cert');

        $conexion = $this->db
            ->where('sucursal_id', $sale['warehouse_id'])
            ->where('activo', 1)
            ->get('sma_clientes_certificador_conexion')
            ->row();

        if (!$conexion) {
            throw new Exception('No hay certificador activo');
        }

        $username    = $conexion->usuario;
        $taxid       = $conexion->nit;
        $certificador = $this->cert->get_certificador($conexion->certificador_id);

        /*
         * ===============================
         * 3️⃣ TOKENS (AMBOS)
         * ===============================
         */
        $tokenFEL = $this->cert->get_token_valido($certificador, $conexion);

        log_message('error', 'TOKEN FEL: ' . $tokenFEL);

        if (!$tokenFEL) {
            throw new Exception('No se pudo obtener token Digifact FEL');
        }

        /*
         * ===============================
         * 4️⃣ CONSULTAR NIT (ANTES DE TODO)
         * ===============================
         */
        if (strtoupper($nit) === 'CF') {
            // Obtener datos del cliente de la base de datos
            $cliente_db = $this->db
                ->select('vat_no, name, address, city as municipio, state as departamento, country as pais')
                ->where('id', $sale['customer_id'])
                ->where('group_name', 'customer')
                ->get('sma_companies')
                ->row_array();

            if (!$cliente_db) {
                throw new Exception('Cliente CF no encontrado en la base de datos');
            }

            $cliente = $cliente_db; // Asignar directamente el array $cliente_db a $cliente
            $cliente['vat_no'] = 'CF'; // Establecer el NIT como CF
        } else {
            $tokenNUC = $this->cert->get_token_valido_nuc($certificador, $conexion);

            if (!$tokenNUC) {
                throw new Exception('No se pudo obtener token Digifact NUC');
            }

            $cliente = $this->consultarNitDigifact($nit, $tokenNUC, $username, $taxid);
            log_message('error', 'CLIENTE OBTENIDO: ' . json_encode($cliente));
        }

        /*
         * ===============================
         * 5️⃣ JSON
         * ===============================
         */
        // Obtener datos del emisor (biller)
        $biller = $this->db
            ->where('group_name', 'biller')
            ->get('sma_companies')
            ->row_array();

         if (!$biller) {
             throw new Exception('Datos del emisor (biller) no encontrados en la base de datos');
         }

        $json = $this->buildJsonDigifact($sale, $items, $cliente, $biller, $tipo);

        /*
         * ===============================
         * 6️⃣ CERTIFICAR
         * ===============================
         */
        if ($modo_prueba) {
            // Devolver el JSON y los datos del cliente para mostrar en el modal
            return [
                'cliente' => $cliente,
                'json' => json_encode($json, JSON_PRETTY_PRINT), // Devolver el JSON formateado
                'reference_no' => $reference_no,
                'taxid' => $taxid,
                'username' => $username,
            ];
        } else {
            $response = $this->sendDigifact($json, $tokenFEL, $taxid, $username);

            if (empty($response['uuid'])) {
                throw new Exception('Certificador no devolvió UUID');
            }

            /*
             * ===============================
             * 7️⃣ GUARDAR
             * ===============================
             */
            $this->saveInvoice($sale, $cliente, $response, $tipo);

            return [
                'cliente' => $cliente,
                'uuid' => $response['uuid'] ?? '',
                'ticket' => $reference_no,
            ];
        }
    }

    /*
     * ==================================================
     * CONSULTA NIT (100% IGUAL A DOC DIGIFACT)
     * ==================================================
     */
    private function consultarNitDigifact($nit, $token, $username, $taxid)
    {
        $url = "https://nucgt.digifact.com/gt.com.apinuc/api/Shared"
            . "?TAXID=" . $taxid
            . "&DATA1=SHARED_GETINFONITcom"
            . "&DATA2=NIT|" . $nit
            . "&COUNTRY=GT"
            . "&USERNAME=" . $username;

        log_message('error', 'INFONIT URL: ' . $url);

        $headers = [
            "Authorization: " . $token, // SIN Bearer (NUC NO USA BEARER)
            "Content-Type: application/json",
        ];

        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTPHEADER => $headers,
        ]);

        $result = curl_exec($ch);

        if ($result === false) {
            log_message('error', 'CURL INFONIT: ' . curl_error($ch));
        }

        curl_close($ch);

        log_message('error', 'INFONIT RESPONSE: ' . $result);

        $data = json_decode($result, true);

        if (empty($data['RESPONSE'])) {
            throw new Exception('NIT no encontrado en SAT');
        }

        if ( ! is_array($data['RESPONSE']) || count($data['RESPONSE']) == 0) {
            throw new Exception('NIT no encontrado en el array de respuesta del SAT');
        }

        $r = $data['RESPONSE'][0]; // Accede al primer elemento del array RESPONSE

        return [
            'vat_no'  => $nit,
            'name'    => $r['NOMBRE'],
            'address' => $r['Direccion'],
            'municipio' => $r['MUNICIPIO'],
            'departamento' => $r['DEPARTAMENTO'],
            'pais' => 'GT' // Asumiendo que el país siempre es Guatemala
        ];
    }

    /*
     * ==================================================
     * CERTIFICAR FEL
     * ==================================================
     */
    private function sendDigifact($json, $token, $taxid, $username)
    {
        $url = 'https://felgtaws.digifact.com.gt/gt.com.apinuc/api/v2/transform/nuc_json'
            . '?TAXID=' . $taxid
            . '&USERNAME=' . $username
            . '&FORMAT=XML';

        $ch = curl_init($url);

        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTPHEADER => [
                "Authorization: " . $token, // SIN Bearer
                "Content-Type: application/json",
            ],
            CURLOPT_POSTFIELDS => json_encode($json),
        ]);

        log_message('error', 'JSON FEL: ' . json_encode($json, JSON_PRETTY_PRINT));

        $result = curl_exec($ch);

        log_message('error', 'FEL RESPONSE (RAW): ' . $result);

        curl_close($ch);

        log_message('error', 'FEL RESPONSE: ' . $result);

        return json_decode($result, true);
    }

private function buildJsonDigifact($sale, $items, $cliente, $biller, $tipo)
{
    $detalle = [];
    $total_impuestos = 0;
    $total_descuentos = 0;
    $gran_total = 0;

    foreach ($items as $i) {
        $item_precio_unitario = floatval($i['unit_price']);
        $item_total = floatval($i['subtotal']);
        $item_impuestos = floatval($i['tax']);
        $item_descuento = floatval($i['discount']);

        $detalle[] = [
            "Number" => count($detalle) + 1,
            "Codes" => null,
            "Type" => "Bien",
            "Description" => $i['product_name'],
            "Qty" => number_format($i['quantity'], 4, '.', ''),
            "UnitOfMeasure" => "UND",
            "Price" => number_format($item_precio_unitario, 2, '.', ''),
            "Discounts" => null,
            "Taxes" => [
                "Tax" => [
                    [
                        "Code" => "1",
                        "Description" => "IVA",
                        "TaxableAmount" => number_format($item_total / 1.12, 2, '.', ''),
                        "Amount" => number_format($item_impuestos, 2, '.', '')
                    ]
                ]
            ],
            "Totals" => [
                "TotalItem" => number_format($item_total, 2, '.', '')
            ]
        ];

        $total_impuestos += $item_impuestos;
        $total_descuentos += $item_descuento;
        $gran_total += $item_total;
    }

    // Codificar el nombre del cliente correctamente
    $nombre_cliente = utf8_encode($cliente['name']);

    // Obtener el código postal del emisor y del receptor
    $codigo_postal_emisor = $biller['postal_code'] ?? '01001'; // Valor por defecto si no se encuentra
    $codigo_postal_receptor = $cliente['postal_code'] ?? '01001'; // Valor por defecto si no se encuentra

    $json = [
        "Version" => "1.00",
        "CountryCode" => "GT",
        "Header" => [
            "DocType" => $tipo,
            "IssuedDateTime" => date('Y-m-d\TH:i:s-06:00', strtotime($sale['date'])),
            "Currency" => "GTQ"
        ],
        "Seller" => [
            "TaxID" => $biller['vat_no'],
            "TaxIDAdditionalInfo" => [
                [
                    "Name" => "AfiliacionIVA",
                    "Value" => "GEN" // O el valor correcto de tu afiliación al IVA
                ]
            ],
            "Name" => $biller['name'],
            "BranchInfo" => [
                "Code" => "1",
                "Name" => $biller['name'],
                "AddressInfo" => [
                    "Address" => $biller['address'],
                    "City" => $biller['city'],
                    "District" => $biller['state'],
                    "State" => $biller['state'],
                    "Country" => $biller['country'],
                    "CodigoPostal" => $codigo_postal_emisor // Añadir el código postal del emisor
                ]
            ]
        ],
        "Buyer" => [
            "TaxID" => $cliente['vat_no'],
            "Name" => $nombre_cliente, // Usar el nombre del cliente codificado
            "AddressInfo" => [
                "Address" => $cliente['address'],
                "City" => $cliente['municipio'],
                "District" => $cliente['departamento'],
                "State" => $cliente['departamento'],
                "Country" => $cliente['pais'],
                "CodigoPostal" => $codigo_postal_receptor // Añadir el código postal del receptor
            ]
        ],
        "Items" => $detalle,
        "Totals" => [
            "TotalTaxes" => [
                "TotalTax" => [
                    [
                        "Description" => "IVA",
                        "Amount" => number_format($total_impuestos, 2, '.', '')
                    ]
                ]
            ],
            "TotalDiscounts" => ($total_descuentos > 0) ? [ "InvoiceDiscount" => number_format($total_descuentos, 2, '.', '') ] : null,
            "GrandTotal" => [
                "InvoiceTotal" => number_format($gran_total, 2, '.', '')
            ]
        ],
        "AdditionalDocumentInfo" => [
            "AdditionalInfo" => [
                [
                    "Code" => "OBS",
                    "Type" => "TEXT",
                    "AditionalData" => [
                        "Data" => [
                            [
                                "Name" => "OBSERVACIONES",
                                "Value" => $sale['note']
                            ],
                            [
                                "Name" => "REFERENCIA_INTERNA",
                                "Value" => $sale['reference_no']
                            ]
                        ]
                    ]
                ]
            ]
        ]
    ];

    return $json;
}
     private function saveInvoice($sale, $cliente, $res, $tipo)
{
    $this->db->insert('sma_fel_invoices', [
        'ticket_id'          => $sale['reference_no'], // Referencia de la venta
        'client_nit'         => $cliente['vat_no'],
        'receptor_nombre'    => $cliente['name'],
        'receptor_direccion' => $cliente['address'],
        'receptor_municipio' => $cliente['municipio'],
        'receptor_departamento' => $cliente['departamento'],
        'receptor_pais'      => $cliente['pais'],
        'uuid'               => $res['uuid'] ?? null, // Asegúrate de que $res['uuid'] exista
        'tipo_dte'           => $tipo,
        'created_at'         => date('Y-m-d H:i:s'),
        // Campos adicionales que podrías obtener de la respuesta de Digifact
        'auth_number'        => $res['authNumber'] ?? null,
        'xml_base64'         => $res['responseData1'] ?? null,
        'html_base64'        => $res['responseData2'] ?? null,
        'pdf_base64'         => $res['responseData3'] ?? null,
        'issued_at'          => $res['issuedTimeStamp'] ?? null,
        'serie'              => $res['serial'] ?? null,
        'numero'             => $res['numero'] ?? null, // Asegúrate de obtener el número correcto
        'company_id'         => $biller['id'] ?? null, // Asumiendo que tienes el ID del emisor
    ]);
}
}